/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.soap;

import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.util.Iterator;

import javax.xml.namespace.QName;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPMessage;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import junit.framework.JUnit4TestAdapter;

import org.picketlink.identity.federation.core.wstrust.SamlCredential;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.junit.Test;

/**
 * Unit test for {@link SOAPSamlHandler}.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * 
 */
public class SOAPSamlHandlerUnitTest
{
    private QName assertionQName = new QName("urn:oasis:names:tc:SAML:2.0:assertion", "Assertion");

    @Test
    public void handleMessage() throws Exception
    {
        final SOAPSamlHandler handler = new SOAPSamlHandler();
        final SOAPMessageContext messageContext = mock(SOAPMessageContext.class);
        final SOAPMessage soapMessage = MessageFactory.newInstance().createMessage();

        when(messageContext.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY)).thenReturn(true);
        when(messageContext.getMessage()).thenReturn(soapMessage);

        SamlCredential samlPrincipal = new SamlCredential(StreamUtils.readStreamString(getClass().getResourceAsStream("assertion.xml"), "UTF-8"));

        boolean result = handler.handleMessage(messageContext);
        assertTrue(result);

        final SOAPEnvelope envelope = soapMessage.getSOAPPart().getEnvelope();
        final Iterator<SOAPElement> securityHeaders = envelope.getHeader().getChildElements(new QName("Security", "wsse", SOAPSamlHandler.WSSE_NS));
        
        while (securityHeaders.hasNext())
        {
            final SOAPElement securityElement = securityHeaders.next();
            final Iterator<SOAPElement> assertions = securityElement.getChildElements(assertionQName);
            while (assertions.hasNext())
            {
                final SOAPElement assertionElement = assertions.next();
                final String expected = samlPrincipal.getAssertionAsString();
                final String actual = SamlCredential.assertionToString(assertionElement);
                assertTrue(XMLHelper.compareXMLContent(expected, actual));
            }
        }
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(SOAPSamlHandlerUnitTest.class);
    }

}
