/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.registry.systinet;

import java.util.Enumeration;

import java.util.Hashtable;
import org.systinet.uddi.client.v2.struct.*;

/**
 * RequestUtil is a utility class which contains a few mappings of UDDI calls :
 * 
 * inquiryQueries maps UDDI request function -> corresponding Systinet request class.
 * publishQueries maps UDDI publication requests functions -> corresponding Systinet request class
 * apiChooser maps request functions -> inquiry/publish API
 * inquiryReturnTypes maps the UDDI inquiry function -> corresponding Systinet response class
 * publishReturnTypes maps the UDDI publish function -> corresponding Systinet response class
 * 
 * @author Tom Cunningham (tcunning@redhat.com)
 */
public class RequestUtil {
	private Hashtable inquiryQueries;
	private Hashtable publishQueries;
	private Hashtable inquiryReturnTypes;
	private Hashtable publishReturnTypes;
	private Hashtable apiChooser;
	
	public static final String INQUIRY_API = "inquiry";
	public static final String PUBLISH_API = "publish";
	
	static private RequestUtil instance = null;
	
	private RequestUtil() {	
		inquiryQueries = new Hashtable();
		inquiryQueries.put("find_business", org.systinet.uddi.client.v2.struct.Find_business.class);
		inquiryQueries.put("find_service", org.systinet.uddi.client.v2.struct.Find_service.class);
		inquiryQueries.put("find_binding", org.systinet.uddi.client.v2.struct.Find_binding.class);
		inquiryQueries.put("find_tmodel", org.systinet.uddi.client.v2.struct.Find_tModel.class);
		inquiryQueries.put("find_relatedbusinesses", org.systinet.uddi.client.v2.struct.Find_relatedBusinesses.class);
		inquiryQueries.put("get_businessdetail", org.systinet.uddi.client.v2.struct.Get_businessDetail.class);
		inquiryQueries.put("get_businessdetailext", org.systinet.uddi.client.v2.struct.Get_businessDetailExt.class);
		inquiryQueries.put("get_servicedetail", org.systinet.uddi.client.v2.struct.Get_serviceDetail.class);
		inquiryQueries.put("get_bindingdetail", org.systinet.uddi.client.v2.struct.Get_bindingDetail.class);
		inquiryQueries.put("get_tmodeldetail", org.systinet.uddi.client.v2.struct.Get_tModelDetail.class);
	
		publishQueries = new Hashtable();
		publishQueries.put("get_authtoken", org.systinet.uddi.client.v2.struct.Get_authToken.class);
		publishQueries.put("get_registeredinfo", org.systinet.uddi.client.v2.struct.Get_registeredInfo.class);
		publishQueries.put("discard_authtoken", org.systinet.uddi.client.v2.struct.Discard_authToken.class);
		publishQueries.put("save_business", org.systinet.uddi.client.v2.struct.Save_business.class);
		publishQueries.put("save_service", org.systinet.uddi.client.v2.struct.Save_service.class);
		publishQueries.put("save_binding", org.systinet.uddi.client.v2.struct.Save_binding.class);
		publishQueries.put("save_tmodel", org.systinet.uddi.client.v2.struct.Save_tModel.class);
		publishQueries.put("delete_business", org.systinet.uddi.client.v2.struct.Delete_business.class);
		publishQueries.put("delete_service", org.systinet.uddi.client.v2.struct.Delete_service.class);
		publishQueries.put("delete_binding", org.systinet.uddi.client.v2.struct.Delete_binding.class);	
		publishQueries.put("delete_tmodel", org.systinet.uddi.client.v2.struct.Find_service.class);
		publishQueries.put("add_publisherassertions", org.systinet.uddi.client.v2.struct.Add_publisherAssertions.class);
		publishQueries.put("set_publisherassertions", org.systinet.uddi.client.v2.struct.Set_publisherAssertions.class);
		publishQueries.put("get_publisherassertions", org.systinet.uddi.client.v2.struct.Get_publisherAssertions.class);
		publishQueries.put("delete_publisherassertions", org.systinet.uddi.client.v2.struct.Delete_publisherAssertions.class);
		publishQueries.put("get_assertionstatusreport", org.systinet.uddi.client.v2.struct.Get_assertionStatusReport.class);

		inquiryReturnTypes = new Hashtable();
		inquiryReturnTypes.put("find_business", org.systinet.uddi.client.v2.struct.BusinessList.class);
		inquiryReturnTypes.put("find_service", org.systinet.uddi.client.v2.struct.ServiceList.class);
		inquiryReturnTypes.put("find_binding", org.systinet.uddi.client.v2.struct.BindingDetail.class);
		inquiryReturnTypes.put("find_tmodel", org.systinet.uddi.client.v2.struct.TModelList.class);
		inquiryReturnTypes.put("find_relatedbusinesses", org.systinet.uddi.client.v2.struct.RelatedBusinessesList.class);
		inquiryReturnTypes.put("get_businessdetail", org.systinet.uddi.client.v2.struct.BusinessDetail.class);
		inquiryReturnTypes.put("get_businessdetailext", org.systinet.uddi.client.v2.struct.BusinessDetailExt.class);
		inquiryReturnTypes.put("get_servicedetail", org.systinet.uddi.client.v2.struct.ServiceDetail.class);
		inquiryReturnTypes.put("get_bindingdetail", org.systinet.uddi.client.v2.struct.BindingDetail.class);
		inquiryReturnTypes.put("get_tmodeldetail", org.systinet.uddi.client.v2.struct.TModelDetail.class);

		publishReturnTypes = new Hashtable();
		publishReturnTypes.put("get_authtoken", org.systinet.uddi.client.v2.struct.AuthToken.class);
		publishReturnTypes.put("get_registeredinfo", org.systinet.uddi.client.v2.struct.RegisteredInfo.class);
		publishReturnTypes.put("discard_authtoken", org.systinet.uddi.client.v2.struct.DispositionReport.class);
		publishReturnTypes.put("save_business", org.systinet.uddi.client.v2.struct.BusinessDetail.class);
		publishReturnTypes.put("save_service", org.systinet.uddi.client.v2.struct.ServiceDetail.class);
		publishReturnTypes.put("save_binding", org.systinet.uddi.client.v2.struct.BindingDetail.class);
		publishReturnTypes.put("save_tmodel", org.systinet.uddi.client.v2.struct.TModelDetail.class);
		publishReturnTypes.put("delete_business", org.systinet.uddi.client.v2.struct.DispositionReport.class);
		publishReturnTypes.put("delete_service", org.systinet.uddi.client.v2.struct.DispositionReport.class);
		publishReturnTypes.put("delete_binding", org.systinet.uddi.client.v2.struct.DispositionReport.class);	
		publishReturnTypes.put("delete_tmodel", org.systinet.uddi.client.v2.struct.DispositionReport.class);
		publishReturnTypes.put("add_publisherassertions", org.systinet.uddi.client.v2.struct.DispositionReport.class);
		publishReturnTypes.put("set_publisherassertions", org.systinet.uddi.client.v2.struct.PublisherAssertions.class);
		publishReturnTypes.put("get_publisherassertions", org.systinet.uddi.client.v2.struct.PublisherAssertions.class);
		publishReturnTypes.put("delete_publisherassertions", org.systinet.uddi.client.v2.struct.DispositionReport.class);
		publishReturnTypes.put("get_assertionstatusreport", org.systinet.uddi.client.v2.struct.AssertionStatusReport.class);

		
		apiChooser = new Hashtable();
		Enumeration e = inquiryQueries.keys();
		while (e.hasMoreElements()) {
			String query = (String) e.nextElement();
			apiChooser.put(query, INQUIRY_API);
		}
		
		e = publishQueries.keys(); 
		while (e.hasMoreElements()) {
			String query = (String) e.nextElement();
			apiChooser.put(query, PUBLISH_API);
		}
	}

	public static RequestUtil getInstance() {
		if (instance == null) {
			instance = new RequestUtil();
		}
		return instance;
	}
	
	public String getApi (String operation) throws Exception {
		return (String) apiChooser.get(operation.toLowerCase());
	}
	
	public Hashtable getInquiryReturnTypes() {
		return inquiryReturnTypes;
	}
	
	public Hashtable getPublishReturnTypes() {
		return publishReturnTypes;
	}
	
	public Hashtable getInquiryQueries() {
		return inquiryQueries;
	}
	
	public Hashtable getPublishQueries() {
		return publishQueries;
	}
}
